import { IParam, PickupView } from "@core";
import { MPickupViewProps } from "./mpickup-view-prop";
import { MPickupViewState } from "./mpickup-view-state";

/**
 * @description 多选视图
 * @export
 * @class MPickupView
 * @extends {PickupView}
 */
export class MPickupView extends PickupView {

  /**
   * @description 视图状态
   * @type {MPickupViewState}
   * @memberof PickupView
   */
  public declare state: MPickupViewState;

  /**
   * @description 视图输入参数
   * @type {MPickupViewProps}
   * @memberof MPickupView
   */
  public declare props: MPickupViewProps;

  /**
   * 初始化选中数据
   *
   * @protected
   * @memberof MPickupView
   */
  protected initSelectedData() {
    const { viewParams } = this.state;
    const { selectedData, selections } = toRefs(this.state);
    selections.value.splice(0, selections.value.length);
    nextTick(() => {
      const selected: IParam = viewParams?.selectedData?.[0] || {};
      const keys = selected.srfkey?.split(',') || [];
      const texts = selected.srfmajortext?.split(',') || [];
      if (keys.length && texts.length) {
        keys.forEach((key: string, index: number) => {
          selectedData.value.push({ srfkey: key, srfmajortext: texts[index] });
        });
      }
      selections.value = [...selectedData.value];
    });
  }

  /**
   * 将选中数据移动到右侧
   *
   * @protected
   * @memberof MPickupView
   */
  protected onMoveRight() {
    const { selections, selectedData } = toRefs(this.state);
    selections.value = [...selectedData.value];
  }

  /**
   * 移动到左侧
   *
   * @protected
   * @memberof MPickupView
   */
  protected onMoveLeft() {
    const { selections, rightPanelSelections } = toRefs(this.state);
    if (rightPanelSelections.value && rightPanelSelections.value.length) {
      for (let i = 0; i < rightPanelSelections.value.length; i++) {
        const index = selections.value.findIndex((select: IParam) => select.srfkey === rightPanelSelections.value[i].srfkey);
        if (index !== -1) {
          selections.value.splice(index, 1);
          //  重新赋值确保数组变化，后续能监听到该变化
          selections.value = [...selections.value];
        }
        rightPanelSelections.value.splice(i, 1);
        i -= 1;
      }
    }
  }

  /**
   * 全部移到右侧（选中）
   *
   * @protected
   * @memberof MPickupView
   */
  protected onAllMoveRight() {
    if (this.xDataControl && this.xDataControl.state) {
      const items = this.xDataControl.state.items || [];
      const { selections, selectedData } = toRefs(this.state);
      selectedData.value = [...items];
      selections.value = [...items];
    }
  }

  /**
   * 全部移到左侧（取消选中）
   *
   * @protected
   * @memberof MPickupView
   */
  protected onAllMoveLeft() {
    const { selections, selectedData } = toRefs(this.state);
    selections.value = [];
    selectedData.value = []; 
  }

  /**
   * 右侧面板项点击
   *
   * @protected
   * @param {IParam} item
   * @param {MouseEvent} event
   * @memberof MPickupView
   */
  protected onRightPanelClick(item: IParam, event: MouseEvent) {
    const { rightPanelSelections } = toRefs(this.state);
    // 判断右侧面板选中数据里面是否有正点击的这项，有就移除，没有就添加
    const index = rightPanelSelections.value.indexOf(item);
    if (index >= 0) {
      rightPanelSelections.value.splice(index, 1);
    } else {
      rightPanelSelections.value.push(item);
    }
  }

  /**
   * 确认
   *
   * @memberof MPickupView
   */
  public onConfirm(): void {
    const { selections } = toRefs(this.state);
    const data: IParam[] = [];
    if (selections.value && selections.value.length) {
      let srfkey: string = '';
      let srfmajortext: string = '';
      selections.value.forEach((select: IParam, index: number) => {
        srfkey += `${select.srfkey}${index !== selections.value.length - 1 ? ',' : ''}`;
        srfmajortext += `${select.srfmajortext}${index !== selections.value.length - 1 ? ',' : ''}`;
      });
      data.push({ srfkey: srfkey, srfmajortext: srfmajortext });
    } else {
      data.push({ srfkey: '', srfmajortext: '' });
    }
    this.emit('viewEvent', { tag: this.state.viewName, action: 'viewDataChange', data: data });
    this.emit('viewEvent', { tag: this.state.viewName, action: 'viewClose', data: null });
  }

  /**
   * @description 安装视图所有功能模块的方法
   * @return {*} 
   * @memberof PickupView
   */
  public moduleInstall() {
    const superParams = super.moduleInstall();
    return {
      ...superParams,
      onMoveRight: this.onMoveRight.bind(this),
      onMoveLeft: this.onMoveLeft.bind(this),
      onAllMoveRight: this.onAllMoveRight.bind(this),
      onAllMoveLeft: this.onAllMoveLeft.bind(this),
      onRightPanelClick: this.onRightPanelClick.bind(this)
    };
  }
}